module net.BurtonRadons.dig.platform.registry;

private import net.BurtonRadons.dig.platform.base;
private import net.BurtonRadons.dig.platform.font;
private import net.BurtonRadons.dig.common.color;


/** A std.math.singleton class for registry access. */
class Registry
{
    /** Store a key in the registry. */
    void saveString (char [] name, char [] data)
    {
        char [] key = "Software\\Dig";
        std.c.windows.windows.HKEY hkey;
        int dot;
        
        while (1)
        {
            dot = std.string.find (name, '.');
            if (dot < 0)
                break;
            
            key ~= "\\" ~ name [0 .. dot];
            name = name [dot + 1 .. name.length];
        }
        
        if (RegCreateKeyA ((_HANDLE) std.c.windows.windows.HKEY_CURRENT_USER, std.string.toStringz (key), &hkey))
            throw new Error ("Couldn't create registry key.");
        if (std.c.windows.windows.RegSetValueExA (hkey, std.string.toStringz (name), 0, REG_SZ, (_LPBYTE) (data ~ "\0"), data.length + 1))
            throw new Error ("Couldn't set registry key.");
        
        std.c.windows.windows.RegCloseKey (hkey);
    }
    
    /** Store a key in the registry. */
    void saveColor (char [] name, Color data)
    {
        char [64] buffer;
        int length;
        
        length = std.c.stdio.sprintf (buffer, "Color: %d, %d, %d, %d", data.r, data.g, data.b, data.a);
        saveString (name, buffer [0 .. length]);
    }
    
    /** Store a key in the registry. */
    void saveInt (char [] name, int value)
    {
        char [64] buffer;
        int length;
        
        length = std.c.stdio.sprintf (buffer, "int: %d", value);
        saveString (name, buffer [0 .. length]);
    }
    
    /** Store a key in the registry. */
    void saveFloat (char [] name, float value)
    {
        char [64] buffer;
        int length;
        
        length = std.c.stdio.sprintf (buffer, "float: %g", (double) value);
        saveString (name, buffer [0 .. length]);
    }
    
    /** Store a font in the registry. */
    void saveFont (char [] name, Font font)
    {
        char [640] buffer;
        int length;
        
        length = std.c.stdio.sprintf (buffer, "font: ");
        for (int c; c < font.digPlatformLog.size; c ++)
            buffer [length + c] = ((char *) &font.digPlatformLog) [c];
        saveString (name, buffer [0 .. length + font.digPlatformLog.size]);
    }
    
    /** Restore a key from the registry. */
    bit loadString (char [] name, out char [] data)
    {
        char [] key = r"Software\Dig";
        char [65535] output;
        int dot, length;
        _DWORD type;
         std.c.windows.windows.HKEY hkey;
        char *ptr;
        
        while (1)
        {
            dot = std.string.find (name, '.');
            if (dot < 0)
                break;
            
            key ~= "\\" ~ name [0 .. dot];
            name = name [dot + 1 .. name.length];
        }
        
        if (RegCreateKeyA ((_HANDLE)  std.c.windows.windows.HKEY_CURRENT_USER, std.string.toStringz (key), &hkey))
            throw new Error ("Couldn't create registry key.");
        length = output.length;
        if (RegQueryValueExA (hkey, std.string.toStringz (name), null, &type, (_LPBYTE) output, (uint *) &length))
            return false;//throw new Error ("Couldn't read registry key.");
        if (type != REG_SZ)
            return false;
        
        if (length && output [length - 1] == 0)
            length -= 1;
        data = output [0 .. length].dup;
        std.c.windows.windows.RegCloseKey (hkey);
        
        return true;
    }
    
    /** Restore a key from the registry. */
    bit loadColor (char [] name, out Color data)
    {
        char [] prefix = "Color: ", s, t;
        int r, g, b, a;
        
        if (!loadString (name, s))
            return false;
        t = s;
        try
        {
            if (s.length < prefix.length || s [0 .. prefix.length] != prefix)
                return false;
            s = s [prefix.length .. s.length];
            while (s.length && !std.ctype.isdigit (s [0]))
                s = s [1 .. s.length];
            if (!digPlatformToInt (s, r)) return false;
            if (!digPlatformToInt (s, g)) return false;
            if (!digPlatformToInt (s, b)) return false;
            if (!digPlatformToInt (s, a)) return false;
            data = SColor (r, g, b, a);
        } finally delete t;
        return true;
    }
    
    /** Restore a key from the registry. */
    bit loadInt (char [] name, out int data)
    {
        char [] prefix = "int: ", s, t;
        
        if (!loadString (name, s))
            return false;
        t = s;
        try
        {
            if (s.length < prefix.length || s [0 .. prefix.length] != prefix)
                return false;
            s = s [prefix.length .. s.length];
            return digPlatformToInt (s, data);
        }
        finally delete t;
        return true;
    }
    
    /** Restore a key from the registry. */
    bit loadFloat (char [] name, out float data)
    {
        char [] prefix = "float: ", s, t;
        
        if (!loadString (name, s))
            return false;
        t = s;
        try
        {
            if (s.length < prefix.length || s [0 .. prefix.length] != prefix)
                return false;
            s = s [prefix.length .. s.length];
            return digPlatformToFloat (s, data);
        }
        finally delete t;
        return true;
    }
    
    /** Restore a font from the registry. */
    bit loadFont (char [] name, out Font data)
    {
        char [] prefix = "font: ", s;
        
        if (!loadString (name, s))
            return false;
        data = new Font (*(LOGFONT *) ((char *) s + prefix.length));
        return true;
    }

    /** Convert an integer to a std.string. */
    static protected bit digPlatformToInt (inout char [] s, out int v)
    {
        int value = 0;
        int sign = 1;
        
        while (s.length && !std.ctype.isdigit (s [0]))
            s = s [1 .. s.length];
        if (!s.length)
            return false;
        
        if (s [0] == '-')
            sign = -1, s = s [1 .. s.length];
        
        while (s.length && std.ctype.isdigit (s [0]))
        {
            value = value * 10 + (s [0] - '0');
            s = s [1 .. s.length];
        }
        
        v = value * sign;
        return true;
    }

    /** Convert a float to a std.string. */
    static protected bit digPlatformToFloat (inout char [] s, out float v)
    {
        real value = 0, num = 0, den = 1;
        int sign = 1;
        
        while (s.length && std.ctype.isspace (s [0]))
            s = s [1 .. s.length];
        if (!s.length)
            return false;
        
        if (s.length >= 3 && s [0 .. 3] == "nan")
        {
            s = s [3 .. s.length];
            v = float.nan;
            return true;
        }
        
        if (s [0] == '-')
            sign = -1, s = s [1 .. s.length];
        else if (s [0] == '+')
            sign = +1, s = s [1 .. s.length];
        
        if (s.length >= 3 && s [0 .. 3] == "inf")
        {
            s = s [3 .. s.length];
            v = float.infinity * sign;
            return true;
        }
        
        while (s.length && std.ctype.isdigit (s [0]))
        {
            value = value * 10 + (s [0] - '0');
            s = s [1 .. s.length];
        }
        
        if (s.length && s [0] == '.')
        {
            s = s [1 .. s.length];
            while (s.length && s [0] >= '0' && s [0] <= '9')
            {
                num = num * 10 + (s [0] - '0');
                den = den * 10;
                s = s [1 .. s.length];
            }
        }
        
        v = (value + num / den) * sign;
        return true;
    }
    
    /** Load or save depending upon the value of load. */
    bit loveInt (char [] name, inout int value, bit load)
    {
        if (load)
            return loadInt (name, value);
        saveInt (name, value);
        return true;
    }
    
    /** Load or save depending upon the value of load. */
    bit loveFloat (char [] name, inout float value, bit load)
    {
        if (load)
            return loadFloat (name, value);
        saveFloat (name, value);
        return true;
    }
    
    /** Load or save depending upon the value of load. */
    bit loveString (char [] name, inout char [] value, bit load)
    {
        if (load)
            return loadString (name, value);
        saveString (name, value);
        return true;
    }
    
    /** Load or save depending upon the value of load. */
    bit loveColor (char [] name, inout Color color, bit load)
    {
        if (load)
            return loadColor (name, color);
        saveColor (name, color);
        return true;
    }
    
    /** Load or save depending upon the value of load. */
    bit loveFont (char [] name, inout Font font, bit load)
    {
        if (load)
            return loadFont (name, font);
        saveFont (name, font);
        return true;
    }
}
